<?php

namespace API\Database\Controllers;

use Slim\App;
use Slim\Routing\RouteCollectorProxy;

use Kreait\Firebase\Factory;

use API\Database\Utils\Functions;
use API\Database\Models\FirebaseModel;
use API\Database\Classes\PreflightAction;
use API\Database\Classes\CommonController;

use Psr\Http\Message\ResponseInterface as Response;
use Psr\Http\Message\ServerRequestInterface as Request;

class FirebaseController extends CommonController
{
    private $factory;
    private $auth;

    public function __construct()
    {
        $this->factory = (new Factory)->withServiceAccount(Functions::getBasePath() . 'server/db-api/assets/akiba-2c9e6-firebase-adminsdk-34b0w-accc4f89c4.json');

        $this->auth = $this->factory->createAuth();

        $this->model = new FirebaseModel();
        $this->routeBase = '/firebase/';
    }

    public function addRoutes(App $app)
    {
        $this->app = $app;

        try {

            $app->group($this->routeBase . 'auth/user', function (RouteCollectorProxy $group) {

                $group->get('', function (Request $request, Response $response) {

                    $this->response = $response;

                    $users = $this->auth->listUsers(10000, 10000);

                    $response->getBody()->write(json_encode($users));

                    return $response->withHeader('content-type', 'application/json')->withStatus(200);
                });

                $group->get('/{uid}', function (Request $request, Response $response) {

                    $this->response = $response;

                    $uid = $request->getAttribute('uid');

                    try {
                        $user = $this->auth->getUser($uid);
                        $response->getBody()->write(json_encode($user));
                    } catch (\Kreait\Firebase\Exception\Auth\UserNotFound $e) {
                        $response->getBody()->write(json_encode(null));
                    }

                    return $response->withHeader('content-type', 'application/json')->withStatus(200);
                });

                $group->post('', function (Request $request, Response $response) {

                    $this->response = $response;

                    $data = $request->getParsedBody();

                    $createdUser = $this->auth->createUser($data);

                    $response->getBody()->write(json_encode($createdUser));

                    return $response->withHeader('content-type', 'application/json')->withStatus(200);
                });

                $group->put('/{uid}', function (Request $request, Response $response) {

                    $this->response = $response;

                    $uid = $request->getAttribute('uid');
                    $data = $request->getParsedBody();

                    try {
                        $updatedUser = $this->auth->updateUser($uid, $data);

                        $response->getBody()->write(json_encode($updatedUser));
                    } catch (\Kreait\Firebase\Exception\Auth\UserNotFound $e) {
                        $response->getBody()->write(json_encode(false));
                    }

                    return $response->withHeader('content-type', 'application/json')->withStatus(200);
                });

                $group->delete('/{uid}', function (Request $request, Response $response) {

                    $this->response = $response;

                    $uid = $request->getAttribute('uid');

                    try {
                        $this->auth->deleteUser($uid);

                        $response->getBody()->write(json_encode($uid));
                    } catch (\Kreait\Firebase\Exception\Auth\UserNotFound $e) {
                        $response->getBody()->write(json_encode($e->getMessage()));
                    } catch (\Kreait\Firebase\Exception\AuthException $e) {
                        $response->getBody()->write(json_encode($e->getMessage()));
                    }

                    return $response->withHeader('content-type', 'application/json')->withStatus(200);
                });
            });

            $app->group($this->routeBase . 'auth/email', function (RouteCollectorProxy $group) {
                $group->post('/verification', function (Request $request, Response $response) {

                    $data = $request->getParsedBody();

                    $actionCodeSettings = [
                        'continueUrl' => isset($data['url']) ? $data['url'] : null,
                    ];

                    $this->auth->sendEmailVerificationLink($data['email'], $actionCodeSettings, isset($data['locale']) ? $data['locale'] : 'fr');

                    $response->getBody()->write(json_encode(true));

                    return $response->withHeader('content-type', 'application/json')->withStatus(200);
                });

                $group->post('/password-reset', function (Request $request, Response $response) {

                    $data = $request->getParsedBody();

                    $actionCodeSettings = [
                        'continueUrl' => isset($data['url']) ? $data['url'] : null,
                    ];

                    $this->auth->sendPasswordResetLink(
                        $data['email'],
                        $actionCodeSettings,
                        isset($data['locale']) ? $data['locale'] : 'fr'
                    );

                    $response->getBody()->write(json_encode(true));

                    return $response->withHeader('content-type', 'application/json')->withStatus(200);
                });

                $group->post('/sign-in', function (Request $request, Response $response) {

                    $data = $request->getParsedBody();

                    $actionCodeSettings = [
                        'continueUrl' => isset($data['url']) ? $data['url'] : null,
                    ];

                    $this->auth->sendSignInWithEmailLink(
                        $data['email'],
                        $actionCodeSettings,
                        isset($data['locale']) ? $data['locale'] : 'fr'
                    );

                    $response->getBody()->write(json_encode(true));

                    return $response->withHeader('content-type', 'application/json')->withStatus(200);
                });
            });
        } catch (\PDOException $e) {

            return $this->errorHandler($e);
        }

        $app->options($this->routeBase . 'auth/user', new PreflightAction());
        $app->options($this->routeBase . 'auth/user/{uid}', new PreflightAction());
        $app->options($this->routeBase . 'auth/email/verification', new PreflightAction());
        $app->options($this->routeBase . 'auth/email/password-reset', new PreflightAction());
        $app->options($this->routeBase . 'auth/email/sign-in', new PreflightAction());
    }
}
